/* Author: Jeff Dalton <J.Dalton@ed.ac.uk> &
 *         Stephen Potter <stephenp@aiai.ed.ac.uk>
 * Last Updated: Thur Sep 19 2002 by Stephen Potter
 * Copyright: (c) 2001-2, AIAI, University of Edinburgh
 */

package ix.jabber;

import java.io.*;

import org.jdom.Element;
import org.jdom.Document;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;
import org.jdom.output.XMLOutputter;

import org.apache.xerces.parsers.*;

import org.w3c.dom.*;
//import com.sun.xml.tree.*;
//import sunlabs.brazil.util.http.*;
import java.util.*;
import java.net.*;

import ix.util.xml.XML;

import ix.util.*;

import org.jabber.jabberbeans.*;
import org.jabber.jabberbeans.Extension.*;
import org.jabber.jabberbeans.serverside.*;
import org.jabber.jabberbeans.util.*;


/*
 * A class that encapsulates the knowledge needed for
 * sending and receiving messages using the Jabber protocol.
 */

public class JabberCommunicationStrategy 
       implements IPC.CommunicationStrategy {

    public ConnectionBean connectionBean;
    public String jabberServer;
    public String username;
    public String resource;
    public String password;
    public String presence;
    public boolean sync; // is communication synchronous or async?
    public IPC.MessageListener ipcListener = null;
    static public JabberWorld jabberworld;

    public JabberCommunicationStrategy() {
    }




    /* for recording the status of the world... */
    public class JabberStatusRecord {
	public String status;
	public JID jid; // using jabberbeans.JID class to store user details
	
	public JabberStatusRecord() {
	}
    
	public JabberStatusRecord(String jn, String st){
	    jid = (new JID(jabberServer)).fromString(jn);
	    status = st;
	    //System.out.println("added: "+jid.toString());
	}
	
	public String getStatus(){
	    return status;
	}

	public void setStatus(String st){
	    status=st;
	}

	public String getJIDString(){
	    return jid.toPrettyString();
	}

	public String getUsername(){
	    return jid.getUsername();
	}

	public String getServer(){
	    return jid.getServer();
	}

	public String getResource(){
	    //  returns null thing if there is no resource specified.
	    return jid.getResource();
	}

	public boolean isJIDEqual(String who){
	    JID jtmp = (new JID(jabberServer)).fromString(who);
	    return jid.equals(jtmp);
	}

	public boolean isOnline(){
	    System.out.println("Is "+jid.toPrettyString()+" on-line? : "+status);
	    if(status.equalsIgnoreCase("Disconnected")){
		return false;
	    }
	    return true;
	}

	public void pprint(){
	    System.out.println(jid.toPrettyString()+" is "+status);
	}
	
    }

    public class JabberWorld {
	// note: the array index here represents the maximum number of 
	// jabber clients that will be stored...
	
	public JabberStatusRecord[] jabberworldarray = new JabberStatusRecord[50];
	public int jwaind = 0;
	public int maxrecords = 50; // make sure this tallies!

	public JabberWorld() {}

	public void addJabberStatusRecord(String who, String what){
	    boolean altered = false;

	    for(int i=0;i<jwaind;i++){
		if(jabberworldarray[i].isJIDEqual(who)){
		    jabberworldarray[i].setStatus(what);
		    altered = true;
		}
	    }

	    if(!altered){
		if(jwaind < (maxrecords-1)){
		    jabberworldarray[jwaind] = new JabberStatusRecord(who,what);
		    jwaind++;
		}
		else
		    System.out.println("Maximum number of Jabber records reached!");
	    }
	}

	public boolean isOnline(String who){
	    /* is who on-line (matching resource names)?
	     * Assumed that who is of form "<user>@<server>/<resource>"
	     */
	    for(int i=0;i<jwaind;i++){
		System.out.println("comparing "+who+ " & "+jabberworldarray[i].getJIDString());
		if(jabberworldarray[i].isJIDEqual(who))
		    return jabberworldarray[i].isOnline();
	    }
	    return false;
	}

	public boolean isIXOnline(String who){
	    /* Does the user specified in who have an I-X resource on-line?
	     * true if yes, false if no.
	     * Assumed that who is of form "<user>@<server>" (with NO 
	     * resource).
	     */
	    for(int i=0;i<jwaind;i++){
		String thisUsername = jabberworldarray[i].getUsername();
		String thisServer = jabberworldarray[i].getServer();
		String res = jabberworldarray[i].getResource();

		if(!(res==null) && res.equalsIgnoreCase("I-X")){ // so this is an IX resource?
		    System.out.println("ix-comparing "+who+" & "+thisUsername+"@"+thisServer);
		    if(who.equalsIgnoreCase(thisUsername+"@"+thisServer))
			return jabberworldarray[i].isOnline();
		}
	    }
	    return false;
	}

	public boolean isUserOnline(String who){
	    /* Does the user specified in who have ANY resource on-line?
	     * true if yes, false if no.
	     * Assumed that who is of form "<user>@<server>" (with NO 
	     * resource).
	     */ 
	    for(int i=0;i<jwaind;i++){
		String thisUsername = jabberworldarray[i].getUsername();
		String thisServer = jabberworldarray[i].getServer();

		System.out.println("user-comparing "+who+ " & "+thisUsername+"@"+thisServer);
		if(who.equalsIgnoreCase(thisUsername+"@"+thisServer)){
		    // note the logic of the following: only want to 
		    // return at this point if resource *is* online; 
		    // otherwise, we still want to check through the user's
		    // other resources...
		    if(jabberworldarray[i].isOnline()) return true;
		}
	    }
	    return false;
	}

	public void pprint(){
	    System.out.println("Number of jabber records="+jwaind);
	    for(int i=0;i<jwaind;i++){
		jabberworldarray[i].pprint();
	    }
	}
    }





    /**
     * Sends an Issue or Report to a named jabber agent by converting
     * it to XML and calling the <code>sendText</code> method.
     */
    public void sendObject(Object destination, Object contents) throws IPC.IPCException {
	String toName = (String)destination;
	boolean sendFail = false; // have the conditions for failure been met?
	JID recipient = (new JID(jabberServer)).fromString(toName);

	/*  The logic here is as follows:
	 * recipient may - or may not - include a resource name.
	 *
	 * if resource name specified:
	 * - if message passing is sync., is resource on-line? if so,
	 * send message, if not fail.
	 * - if message passing is async, send message.
	 *
	 * if no resource name: 
	 * - if recipient is the user of this panel, add "I-X" resource,
	 * and send message.
	 * - else if the recipient has an "I-X" resource on-line, then 
	 * assume this is this the target: add resource & send.
	 * - else if sync. message passing and the recipient has *some*
	 * resource on-line, send message (with null resource - let the
	 * jabber server sort it out).
	 */

	System.out.println("Request to send message to: "+toName);

	if(!(recipient.getResource()==null)){ // ie a resource *is* specified
	    if(sync && !jabberworld.isOnline(toName)){
		// if synchronous message passing is specified and
		// the recipient is *not* on-line:
		sendFail = true;
		System.out.println(toName+ ": resource is not on-line: message not sent.");
	    }
	}
	else{ // if no resource is specified...

	    System.out.println("No resource specified.");

	    if(toName.equalsIgnoreCase(username+"@"+jabberServer)){
		// is the recipient the user of this panel?
		// if so, assume this panel is the target...
		toName = toName+"/"+resource; // ...and add resource
	    }
	    else if(jabberworld.isIXOnline(toName)){
		// if recipient is not this panel, and no resource is 
		// specified and the ix panel of the recipient user *is* 
		// on-line, then assume that that panel is the intended 
		// recipient...
		toName = toName+"/"+resource; // ...and add resource
		System.out.println(toName+" is on-line: assuming this is intended recipient.");
	    }
	    else if(sync && !jabberworld.isUserOnline(toName)){
		// else, if recipient is not this panel, and no resource
		// is specified, synchronous messaging is specified and 
		// the recipient user has *no* on-line resources:
		sendFail = true;
		System.out.println(toName+ " has no on-line resources: message not sent.");
	    }
	}

	if(sendFail){
	    System.out.println("Failure to send to: "+toName);
	    throw new IPC.IPCException("Recipient "+toName+" is not on-line (or you have not subscribed to this user's presence)\nMessage not sent");
	}
	else{
	    sendText(toName, XML.objectToXMLString(contents));
	}
    }


    /**
     * Sends a string to a named I-X panel.
     */
    public void sendText(String toName, String contents){


	MessageBuilder mb=new MessageBuilder();
	Message msg;
	//set destination user.

	// assume that toName is of the form username@jabberserver/resource...

	Debug.noteln("\nSending to:"+toName);

	mb.setToAddress(new JID(toName));
	
	// want to inspect the contents to see whether this message is
	// a 'chat' message...:
	try{
	    String SAXDriverClass = "org.apache.xerces.parsers.SAXParser";
	    
	    StringReader reader = new StringReader(contents);
	    SAXBuilder builder = new org.jdom.input.SAXBuilder(SAXDriverClass);
	    Document doc = builder.build(reader);
	    //System.out.println("document= "+ doc.toString());
	    
	    Element ixroot = doc.getRootElement();
	    //System.out.println("ixroot= "+ ixroot.getName());

	    // if we have an I-X chat-message, we now want to translate
	    // it into the equivalent Jabber message of type "chat":
	    if((ixroot.getName()).equalsIgnoreCase("chat-message")){
		// want to extract the content of this and wrap as jabber-chat:		
		// ix xml parsing bit:
		Element ixtext = ixroot.getChild("text");
		//System.out.println("child= "+ ixpattern.toString());
		Element ixstring = ixtext.getChild("string");
		//System.out.println("textcont= "+ ixstring.getText());

		// For jabber, just set contents to be the text of the 
		// string element:
		contents = ixstring.getText();
		// and need to set the type appropriately:
		mb.setType("chat");
		
	    }

	}
	catch(Exception e){
	    System.out.println(e.toString());
	}



	mb.setBody(contents);
	try
	    {
		msg=(Message)mb.build();
	    }
	catch (InstantiationException e)
	    {
		//build failed ?
		System.out.println("error creating message packet:");
		System.out.println(e.toString());
		return;
	    }

	//send message
	connectionBean.send(msg);
    }


    // logon to the named jabber server...
    public void logon(){
	
	InfoQueryBuilder iqb=new InfoQueryBuilder();
	InfoQuery iq;
	//and the auth data builder
	IQAuthExtensionBuilder iqAuthb=new IQAuthExtensionBuilder();

	//we are setting info
	iqb.setType("set");

	//user,password & resource are from the parameter list (or defaults).
	iqAuthb.setUsername(username);
	iqAuthb.setPassword(password);
	iqAuthb.setResource(resource);

	//build the Auth data
	try
	{
	    iqb.addExtension(iqAuthb.build());
	}
	catch (InstantiationException e)
	{
	    //building failed ?
	    java.lang.System.out.println("Fatal Error on Auth object build:");
	    java.lang.System.out.println(e.toString());
	    return;
	}

	//build the iq packet
	try
	{
	    //build the full InfoQuery packet
	    iq=(InfoQuery)iqb.build();
	}
	catch (InstantiationException e)
	{
	    //building failed ?
	    java.lang.System.out.println("Fatal Error on IQ object build:");
	    java.lang.System.out.println(e.toString());
	    return ;
	}

	//send the InfoQuery packet to the server to log in
	connectionBean.send(iq);
	
	java.lang.System.out.println("Send requested for IQ");
	
    }


    /**
     * Initiate a jabber receiver object and set up for receiving messages.
     */
    public void setupServer(Object destination, 
			    IPC.MessageListener listener) {

	InetAddress host;
	connectionBean = new ConnectionBean();
	ipcListener = listener;
	jabberworld = new JabberWorld();

	if(Parameters.haveParameter("jabber-server")){
	    jabberServer = Parameters.getParameter("jabber-server");
	}
	else
	    jabberServer = "akt.aiai.ed.ac.uk"; // default

	if(Parameters.haveParameter("jabber-username")){
	    username = Parameters.getParameter("jabber-username");
	}
	else
	    username = "IX-test"; // default

	if(Parameters.haveParameter("jabber-password")){
	    password = Parameters.getParameter("jabber-password");
	}
	else
	    password = "jabber"; // default

	if(Parameters.haveParameter("jabber-resource")){
	    resource = Parameters.getParameter("jabber-resource");
	    // don't allow a null resource name:
	    if(resource.equalsIgnoreCase("")) resource = "I-X";
	}
	else
	    resource = "I-X"; //default

	if(Parameters.haveParameter("jabber-allow-queuing")){
	    
	    if((Parameters.getParameter("jabber-allow-queuing")).equalsIgnoreCase("true")) sync = false;
	    else sync = true;
	}
	else
	    sync = true; // default: synchronous message-passing

	if(sync==true) System.out.println("Synchronous messaging selected.");
	else System.out.println("Asynchronous messaging selected.");


	try{
	    connectionBean.addPacketListener(new MyPacketListener());
	    connectionBean.addConnectionListener(new MyConnectionListener());

	    host = InetAddress.getByName(jabberServer);
	    System.out.print("got host\n");
	    connectionBean.connect(host);
	    System.out.print("connected!\n");	    
	    logon();
	}
	catch (Exception e){
	    System.out.print(e);
	}

	PresenceBuilder pb=new PresenceBuilder();
	    
	//this is presence to the irc transport on jabber.org. This will log
	//you into the #jabber channel. The best way to see this working is to
	//be on the jabber channel yourself (it is on irc.openprojects.net)
	//	JID dest=new JID("#jabberbeans","irc.jabber.org",
	//		"jtest");
	//pb.setToAddress(dest);

	if(Parameters.haveParameter("jabber-presence")){
	    presence = Parameters.getParameter("jabber-presence");
	    
	    // in case of empty/null parameter declaration:
	    if(presence.equalsIgnoreCase("")) presence = "Online";
	}
	else
	    presence = "Online"; // default

	String show = "";

	if(presence.equalsIgnoreCase("Away") || 
	   presence.equalsIgnoreCase("Lunch") ||
	   presence.equalsIgnoreCase("Bank"))
	    show = "away";
	if(presence.equalsIgnoreCase("Busy") || 
	   presence.equalsIgnoreCase("Working") ||
	   presence.equalsIgnoreCase("Mad"))
	    show = "dnd";
	if(presence.equalsIgnoreCase("Extended Away") || 
	   presence.equalsIgnoreCase("Gone Home") ||
	   presence.equalsIgnoreCase("Gone to Work") ||
	   presence.equalsIgnoreCase("Sleeping"))
	    show = "xa";
	if(presence.equalsIgnoreCase("Wants to Chat"))
	    show = "chat";

	pb.setStatus(presence);
	pb.setStateShow(show);

	try
	    {
		//build and send presence
		connectionBean.send(pb.build());
	    }
	catch (InstantiationException e)
	    {
		//building failed ?
		System.out.println
		    ("Fatal Error on Presence object build:");
		System.out.println(e.toString());
		return ;
	    }
    
    }


    static class MyConnectionListener implements ConnectionListener
    {
	public void connected(ConnectionEvent ce)
	{
	    System.out.println("CLDebug: Connected!");
	}
	public void disconnected(ConnectionEvent ce)
	{
	    System.out.println("CLDebug: Disconnected");
	}
	public void connecting(ConnectionEvent ce)
	{
	    System.out.println("CLDebug: Connecting...");
	    }
	public void connectFailed(ConnectionEvent ce)
	{
	    System.out.println("CLDebug: Connect Failed");
	}
	public void connectionChanged(ConnectionEvent ce)
	{
	    System.out.println("CLDebug: Connection Changed");
	}
    }


    class MyPacketListener implements PacketListener {

	public void receivedPacket(PacketEvent pe)
	{
	    Packet p = pe.getPacket();

	    System.out.println("PLDebug: RECEIVED PACKET!!! : " 
			       + p.toString());

	    if(p instanceof Message){

		try{
		    Message m = (Message)p;
	
		    // normal, chat, or headline
		    String mtype = m.getType();
		    if(mtype==null) mtype="null";
		    String from = m.getFromAddress().toString();
		    String mbody = m.getBody();

		    System.out.println("Message of type: " + mtype);
		    System.out.println("\treceived from: " + from);
		    System.out.println("\tbody: " + mbody);

		    Object contents;

		    boolean ixmessage = false;

		    // want to inspect mbody, to see whether it is valid
		    // IX-XML...
		    try{
			String SAXDriverClass = "org.apache.xerces.parsers.SAXParser";
			StringReader reader = new StringReader(mbody);
			SAXBuilder builder = new org.jdom.input.SAXBuilder(SAXDriverClass);
			Document doc = builder.build(reader);
			String ixrootname = (doc.getRootElement()).getName();
			
			// System.out.println("ixroot= "+ ixrootname);
			// is this an ixmessage?
			// these are all? the valid types of ix message:
			if(ixrootname.equalsIgnoreCase("issue") ||
			   ixrootname.equalsIgnoreCase("activity") ||
			   ixrootname.equalsIgnoreCase("constraint") ||
			   ixrootname.equalsIgnoreCase("report") ||
			   ixrootname.equalsIgnoreCase("chat-message")){
			    ixmessage = true;
			    System.out.println("Recognised I-X XML message.");
			}
			// else? not sure what to do if a bit of random
			// xml is sent??

		    }
		    catch(Exception e){
			//System.out.println(e.toString());
			System.out.println("Incoming message is not a structured I-X XML message: converting to chat-message.");
		    }


		    // if this is a jabber chat message, must translate it
		    // (back!) into the equivalent I-X xml form:
		    if(mtype.equalsIgnoreCase("chat") || ixmessage==false){
			contents = XML.objectFromXML("<?xml version=\"1.0\" encoding=\"UTF-8\"?><chat-message sender-id=\""+from+"\"><text><string>"+mbody+"</string></text></chat-message>");
		    }
		       else
			   contents = XML.objectFromXML(mbody);

		    if(mtype.equalsIgnoreCase("error")) 
			// if there is an error, do nothing (for now).
			System.out.println("Error message received (remote server error?).");
		    else
			// else notify the ipcListener:
			ipcListener.messageReceived(new IPC.BasicInputMessage(contents));
		}
		catch (Exception e){

		    Debug.noteException(e);

		}

	    }

	    if(p instanceof Presence){
		try{
		    Presence pr = (Presence)p;
		    String from = pr.getFromAddress().toString();
		    String status = pr.getStatus();

		    if ((!(pr.getType()==null)) 
			&& ((pr.getType()).equalsIgnoreCase("error"))){
			/* if there is an error (of whatever type) in the
			 * presence message, assume that the referred resource
			 * is Disconnected.
			 */
			status = "Disconnected";
			System.out.println("\nPresence error noted - assuming:");
		    }

		    System.out.println("\nPresence: " 
				       + from 
				       + " --- " + status + "\n");

	
		    jabberworld.addJabberStatusRecord(from,status);
		    jabberworld.pprint();

		    Object contents = XML.objectFromXML("<?xml version=\"1.0\" encoding=\"UTF-8\"?><constraint type=\"world-state\" relation=\"effect\" sender-id=\""+from+"\"><parameters><list><pattern-assignment><pattern><list><symbol>presence</symbol><symbol>"+from+"</symbol></list></pattern><value><symbol>"+status+"</symbol></value></pattern-assignment></list></parameters></constraint>");

		    ipcListener.messageReceived(new IPC.BasicInputMessage(contents));

		}
		catch (Exception e){
		    Debug.noteException(e);
		}
	    }		
	}

	public void sentPacket(PacketEvent pe)
	{
	    System.out.println("PLDebug: Sent Packet : " + 
			       pe.getPacket().toString());
	}
	public void sendFailed(PacketEvent pe)
	{
	    System.out.println("PLDebug: Send Failed Packet : " + 
			       pe.getPacket().toString());
	}
    }

}
